%
%   statistical_results_2.m
%
%   look at grilli and kirby error estimates for the various models.
%
%   This  version plots all of the error estimates for a given model, with
%   the models classed as hydrodynamic types.
%
%   Models are categorized in terms of 4 hydrodynamic characterizations:

%   1 - hydrostatic
%   2 - weakly dispersive
%   3 - nonhydrostatic
%   4 - Navier-Stokes solvers

%   and 4 slide configurations:
%
%   1 - solid slide or analytic bottom motion
%   2 - viscous layer
%   3 - granular layer
%   4 - 3D multiphase
%
%   Error measures are:
%
%   1 - kirby - sqrt( sum(sum( (false-true)^2 ))/sum(sum( true^2 )) )
%   2 - grilli - sqrt( sum( false-true)^2 ))/(max(true)-min(true)), each
%                gage
%
%
%   dat file: error_data.mat
%
%   columns: 1=model, 2=hydro type, 3=slide type,
%            4-7=grilli error, bm2,d=61, 8=kirby error, same
%            9-12=grilli error, bm2,d=120, 13=kirby error, same
%            14-17=grilli error, bm4, 18=kirby error, same
%
%   J. Kirby, 1/7/17,1/17/18
%--------------------------------------------------------------------------

disp('List of available benchmarks')

   disp('1: Benchmark 2, Solid slide, d=061');
   disp('2: Benchmark 2, Solid slide, d=120');
   disp('3: Benchmark 4, Submarine granular slide');
   
disp('model identifiers');

   disp('0: benchmark data');
   disp('1: NHWAVE: 3D non-hydrostatic model, solid slide (UD, ODU, URI)');
   disp('2: NHWAVE: 3D non-hydrostatic model, viscous fluid slide (UD, UA)');
   disp('3: NHWAVE: 3D non-hydrostatic model, granular slide');
   disp('4: GloBous: Boussinesq model, linear');
   disp('5: GloBous: Boussinesq model, nonlinear');
   disp('6: BousClaw: standard dispersion');
   disp('7: BousClaw: pade dispersion');
   disp('8: TSUNAMI3D: Navier-Stokes (TAMUG)');
   disp('9: THETIS: Navier Stokes'); 
   disp('10: NHWAVE: 3D non-hydrostatic model, vontinuous density field');
   disp('11: THETIS: mu(I) viscosity');
   disp('12: L3SD');
   disp('13: 2LCMFLOW');
   disp('14: Alaska GI-L');
   disp('15: NHWAVE-solid, hydrostatic');
   disp('16: NHWAVE-granular, hydrostatic');
   disp('17: Landslide-hysea, non-hydrostatic');
   disp('18: fbslide, hydrostatic');
   disp('19: NLSWE (Lynett)');
   disp('21: MSE (Lynett)');
   
disp('Model hydrodynamic types');

   disp('1 - hydrostatic');
   disp('2 - weakly dispersive');
   disp('3 - nonhydrostatic');
   disp('4 - Navier-Stokes solvers');
   
disp('Slide types');
   disp('1 - solid slide or analytic bottom motion');
   disp('2 - viscous layer');
   disp('3 - granular layer');
   disp('4 - 3D multiphase');
   
%  specify which benchmarks to consider

   bmark=input('Enter list of benchmarks in square brackets: ');

%  specify which hydrodynamic model types to consider

%   htype=input('enter list of hydrodynamic model types in square brackets: ');
   
%  specify which slide types to consider

   stype=input('Enter list of slide types in square brackets: ');
   
%  number of benchmarks

   nb=length(bmark);
   
%  number of hydrodynamic types

%   nh=length(htype);
   
%  number of slide types

   ns=length(stype);
   
%  when plotting errors, we need to specify symbols to identify model and
%  slide types.  For hydro-oriented plots, I'm going to use:
%           o=1
%           p=2
%           h=3  hexagram
%           *=4
%
   hsym=char('o','p','h','*');
%   
%  Slide types are going to be identified by color:
%           r=solid
%           g=viscous
%           b=granular
%           k=3D
%
   ssym=char('r','g','b','k');
%
%  define which columns in the error_data array are associated with which
%  gages, with entries 1,2,3 corresponding to benchmarks 1,2,3.

   g1_cols=[4 9 14]; 
   g2_cols=[5 10 15]; 
   g3_cols=[6 11 16]; 
   g4_cols=[7 12 17];
   kirby_cols=[8 13 18];

   load error_data
   
   models=error_data(:,1:3); s=size(models); nmods=s(1);
   
   figure(1),clf
   %  This plot is error as a function of model number
   
   imod=0; lim1=0.75; lim2=nmods+0.25;
  
   plot(5,5,'x'), axis([lim1 lim2 0 .4]), hold on
   set(gca,'xtick',1:nmods);
   ax=gca;
   ax.FontSize = 15;
   ax.FontName = 'Times';
   xlabel('Model number', 'interpreter', 'latex', 'fontsize', 20)
   ylabel('RMS error measure', 'interpreter', 'latex', 'fontsize', 20)
   
   %  Outer loop is over hydrodynamic model types
   for ih=1:4,
       %  loop over all models and identify whether they are of this model
       %  type
       for mindex=1:nmods,
           if models(mindex,2)==ih,
               %  Do stuff for this model
               imod=imod+1;
               s=models(mindex,3);
               sym=[hsym(ih), ssym(s)];
               ib=0; is=0;
               %  Next, loop over benchmarks
               for ib=1:3,
                  bm=bmark(ib);
                  g1=g1_cols(bm); g2=g2_cols(bm); g3=g3_cols(bm); g4=g4_cols(bm);           
                  %  and over slide types
                  for is=1:ns,
                    % Is the slide type for this model in the list
                    % of desired types?  If so, plot symbol for this model
                    if any(s==stype),
  
                       if error_data(mindex,g1)~=0.,
                         plot(imod, error_data(mindex,g1), sym, 'MarkerSize', 15)
                       end
        
                       if error_data(mindex,g2)~=0.,
                         plot(imod, error_data(mindex,g2), sym, 'MarkerSize', 15)
                       end
 
                       if error_data(mindex,g3)~=0.,                      
                         plot(imod, error_data(mindex,g3), sym, 'MarkerSize', 15)
                       end

                       if error_data(mindex,g4)~=0.,                        
                         plot(imod, error_data(mindex,g4), sym, 'MarkerSize', 15)
                       end
                    end
                  end
               end
           end
       end
   end
   
   print -dpng stats2.png
   
   
            
               
